/******************************************************************************
 * %Id: int_dev.c 227 2010-02-26 05:50:39Z 4015169 $
 *
 * FileName     :int_drv.c
 *
 * Description  :Common Driver (Common part)
 *
 *
 * Copyright	:Panasonic Corporation.
 *
 *****************************************************************************/

/* ==== Attention ================================================== */
/* This module doesn't support fork() nor dup() system call          */
/* which copies (duplicates) memory space or file descripter.        */
/* ================================================================= */

#undef  INTDEV_VERBOSE
#undef  INTDEV_LOG_WAKEUP

#ifndef INT_DEV_C
#define INT_DEV_C
#define __NO_VERSION__ /* Must be first in all but one driver source file! */

#include <linux/autoconf.h>
#ifdef MODULE
#include <linux/module.h>  /* Definitions needed for kernel modules */
#endif
#include <linux/kernel.h>  /* We run in the kernel so we need this */
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/rwsem.h>
#include <linux/rwsem-spinlock.h>
#include <asm/uaccess.h>
#include <asm/delay.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include "dev-share.h"
#include <asm/hw_irq.h>
#include <linux/iosc/int_dev.h>
#include <arch/arm/mach-uniphier/include/mach/intctl-regs.h>

#define ERR(fmt, args...) printk("[int_dev] " fmt "\n", ##args)
#ifdef INTDEV_VERBOSE
# define MSG(fmt, args...) printk("{int_dev} " fmt "\n", ##args)
#else
# define MSG(...) /* empty */
#endif

#define INT_DEV_MIN_PORT_INTNO (80)
#define INT_DEV_MAX_PORT_INTNO (95)

#define IS_EXTIRQ(num) ((num)>=INT_DEV_MIN_PORT_INTNO && (num)<=INT_DEV_MAX_PORT_INTNO)

#define IRQCTL		__SYSREG(0x55000090)	/* external irq control */

/* Interrupt level (int_dev uses most lowest level in Linux region). */
#define INT_DEV_ICR_LEVEL 6
#define GxICR_LPINT 0x00020000

typedef struct _intr_info_t {
  char name[10];
  int enable;
  int receive_flag;
  void *fd_info;
  wait_queue_head_t intr_wait_queue;
} intr_info_t;

intr_info_t *intr_info;

static spinlock_t intdev_lock = __SPIN_LOCK_UNLOCKED(intdev_lock);
static struct semaphore intdev_sem;
static unsigned long irqctl;

enum intdev_factor {
  intdev_factor_low = 0,
  intdev_factor_high,
  intdev_factor_neg,
  intdev_factor_pos,
};

/* local function */
static int intdev_local_open( struct inode *inode, struct file *filp );
static int intdev_local_release( struct inode *inode, struct file *filp );
static int intdev_local_ioctl( struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg );
static irqreturn_t intdev_interrupt( int irq, void *dev_id );


#define MAJOR_MINOR_CHECK(ma, mi) \
  do {                           \
    if (ma != INT_DEV_MAJOR) {  \
      return -EPERM;        \
    }                         \
    if (mi >= INT_DEV_MINOR_NUM) {          \
      return -ENOENT;                     \
    }                                       \
  } while(0)
#define ID_CHECK(id)                            \
  do {                                         \
    if (id >= INT_DEV_OPEN_MAX) {             \
      return -EPERM;                      \
    }                                       \
  } while(0)

#define BIT_ISSET(pattern, flag) ((flag)&(pattern) ? TRUE : FALSE)
#define BIT_SET(pattern, flag)   ((flag) |= (pattern))
#define BIT_CLR(pattern, flag)   ((flag) &= ~(pattern))
#define BIT_ZERO(flag)         ((flag) = 0)

#define INTDEV_EVENT_CLOSE (0x1<<0)
#define INTDEV_EVENT_INTR  (0x1<<1)
#define INTDEV_EVENT_WAKEUP  (0x1<<2)

#define INTDEV_STATE_USER_ENABLE (0x1<<0)
#define INTDEV_STATE_INTR_DISABLE   (0x1<<1)



/*****************************************************************
 * Method functions for Linux-side device
 *****************************************************************/

/*
 * [Method] open
 * RETURN: 0
 */
static struct semaphore intdev_sem = __SEMAPHORE_INITIALIZER(intdev_sem, 1);
static int intdev_local_open(struct inode *inode, struct file *filp)
{
  int major = MAJOR( inode->i_rdev );
  int minor = MINOR( inode->i_rdev );

  MAJOR_MINOR_CHECK(major,minor);

  /* do nothing */

  return 0;
}

/*
 * [Method] close
 * RETURN: 0
 */
static int intdev_local_release(struct inode *inode, struct file *filp)
{
  int i;
  int semret;
  unsigned long flags;

  semret = down_interruptible(&intdev_sem);
  
  for (i=0; i<NR_IRQS; i++) {
    intr_info_t *info = &intr_info[i];

    /* Used filep to relate fd which destroy INTR.  */
    /* Because the address of filp is filed each fd */
    if (info->fd_info == filp) {
      MSG("release (num=%d, filp=0x%08x)", i, (unsigned int)filp);
      disable_irq(i);
      spin_lock_irqsave(&intdev_lock, flags);
      BIT_SET(INTDEV_EVENT_CLOSE, info->receive_flag);
      wake_up_interruptible(&info->intr_wait_queue);
      spin_unlock_irqrestore(&intdev_lock, flags);
      free_irq(i, (void *)info);
      info->fd_info = NULL;
    }
  }

  up(&intdev_sem);

  /* FIXME: disable and release interrupts used by this tgid */

  return 0;
}

static int intdev_local_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
  int int_num, factor;
  int semret;
  int ret = 0;
  intr_info_t *info;
  unsigned long flags;

  /* parameter check */
  int_num = (int)arg;
  if (int_num < 0 || NR_IRQS <= int_num) {
    return -EPERM;
  }

  info = &intr_info[int_num];

  switch (cmd) {
  case INT_IOC_CREATE_INT:
    /* Exclusion with the interruption handler is unnecessary.  */
    /* because interruption don't enable. */
    MSG("INT_IOC_CREATE_INT (%d)", int_num);
    semret = down_interruptible(&intdev_sem);   /* ++++ */
    if (info->fd_info != NULL) {
      up(&intdev_sem);                 /* ---- */
      return -EBUSY;
    }
    mn_intc_set_level(int_num, INT_DEV_ICR_LEVEL);

    if (IS_EXTIRQ(int_num)) {
      unsigned int val = readl(irqctl);
      val |= (0x00000001 << (int_num - INT_DEV_MIN_PORT_INTNO));
      writel(val, irqctl);
    }

    ret = request_irq(int_num, intdev_interrupt, IRQF_DISABLED, info->name, (void*)info);
    if (ret < 0) {
      up(&intdev_sem);                 /* ---- */
      return ret;
    }

    BIT_ZERO(info->enable);
    BIT_ZERO(info->receive_flag);
    init_waitqueue_head(&(info->intr_wait_queue));
    info->fd_info = filp;
    up(&intdev_sem);                   /* ---- */
    return 0;
    break;

  case INT_IOC_DESTROY_INT:
    MSG("INT_IOC_DESTROY_INT (%d)", int_num);
    semret = down_interruptible(&intdev_sem);   /* ++++ */
    if (info->fd_info == NULL) {
      up(&intdev_sem);                 /* ---- */
      return -ENOENT;
    }
    disable_irq(int_num);
    
    spin_lock_irqsave(&intdev_lock, flags);
    BIT_SET(INTDEV_EVENT_CLOSE, info->receive_flag);
    wake_up_interruptible(&info->intr_wait_queue);
    spin_unlock_irqrestore(&intdev_lock, flags);
    
    free_irq(int_num, (void*)info);
    
    /* initialize infomation */
    info->fd_info = NULL;
    up(&intdev_sem);                 /* ---- */
    return 0;
    break;

  case INT_IOC_FACTOR_LOW:
    MSG("INT_IOC_FACTOR_LOW (%d)", int_num);
    factor = XIRQ_TRIGGER_LOWLEVEL;
    goto change_factor_label;

  case INT_IOC_FACTOR_HIGH:
    MSG("INT_IOC_FACTOR_HIGH (%d)", int_num);
    factor = XIRQ_TRIGGER_HILEVEL;
    goto change_factor_label;

  case INT_IOC_FACTOR_NEG:
    MSG("INT_IOC_FACTOR_NEG (%d)", int_num);
    factor = XIRQ_TRIGGER_NEGEDGE;
    goto change_factor_label;

  case INT_IOC_FACTOR_POS:
    MSG("INT_IOC_FACTOR_POS (%d)", int_num);
    factor = XIRQ_TRIGGER_POSEDGE;
    goto change_factor_label;

  change_factor_label:
    if (!IS_EXTIRQ(int_num))
      return -EINVAL;
    /* To cahnge factor, prohibit interrupt.  */
    spin_lock_irqsave(&intdev_lock, flags);
    SET_XIRQ_TRIGGER(int_num - INT_DEV_MIN_PORT_INTNO, factor);
    spin_unlock_irqrestore(&intdev_lock, flags);
    return 0;

  case INT_IOC_ENABLE_INT:
    MSG("INT_IOC_ENABLE_INT (%d)", int_num);
    semret = down_interruptible(&intdev_sem);                 /* ++++ */
    if (info->fd_info == NULL) {
      up(&intdev_sem);                 /* ---- */
      return -ENOENT;
    }
    if (info->fd_info != filp) {
      up(&intdev_sem);                 /* ---- */
      return -EPERM;
    }

    spin_lock_irqsave(&intdev_lock, flags);
    if (!BIT_ISSET(INTDEV_STATE_USER_ENABLE, info->enable)) {
      BIT_SET(INTDEV_STATE_USER_ENABLE, info->enable);
      BIT_ZERO(info->receive_flag);
      enable_irq(int_num);
    }
    spin_unlock_irqrestore(&intdev_lock, flags);
    up(&intdev_sem);                 /* ---- */
    MSG("G%dICR=0x%08x", int_num, inl(GxICR(int_num)));
    return 0;
    break;
    
  case INT_IOC_DISABLE_INT:
    MSG("INT_IOC_DISABLE_INT (%d)", int_num);
    semret = down_interruptible(&intdev_sem);                 /* ++++ */
    if (info->fd_info == NULL) {
      up(&intdev_sem);                 /* ---- */
      return -ENOENT;
    }
    if (info->fd_info != filp) {
      up(&intdev_sem);                 /* ---- */
      return -EPERM;
    }

    spin_lock_irqsave(&intdev_lock, flags);
    if (BIT_ISSET(INTDEV_STATE_USER_ENABLE, info->enable)){
      if (!BIT_ISSET(INTDEV_STATE_INTR_DISABLE, info->enable)){
        disable_irq_nosync(int_num);
      }
      BIT_ZERO(info->enable);
    }
    spin_unlock_irqrestore(&intdev_lock, flags);
    up(&intdev_sem);                  /* ---- */
    MSG("G%dICR=0x%08x", int_num, inl(GxICR(int_num)));
    return 0;
    break;

  case INT_IOC_WAIT_INT:
    MSG("INT_IOC_WAIT_INT (%d)", int_num);
    MSG("G%dICR = 0x%08x", int_num, inl(GxICR(int_num)));
    semret = down_interruptible(&intdev_sem);                 /* ++++ */
    if (info->fd_info == NULL) {
      up(&intdev_sem);                 /* ---- */
      return -ENOENT;
    }
    if (info->fd_info != filp) {
      up(&intdev_sem);                 /* ---- */
      return -EPERM;
    }
    
    spin_lock_irqsave(&intdev_lock, flags);
    if (BIT_ISSET(INTDEV_EVENT_WAKEUP, info->receive_flag)){
      BIT_CLR(INTDEV_EVENT_INTR|INTDEV_EVENT_WAKEUP, info->receive_flag);
      BIT_CLR(INTDEV_STATE_INTR_DISABLE, info->enable);
      if(BIT_ISSET(INTDEV_STATE_USER_ENABLE, info->enable)){
        enable_irq(int_num);
      }
    }
    spin_unlock_irqrestore(&intdev_lock, flags);
    up(&intdev_sem);                 /* ---- */
    
    ret = wait_event_interruptible(info->intr_wait_queue, info->receive_flag != 0);
    BIT_SET(INTDEV_EVENT_WAKEUP, info->receive_flag);
    
#ifdef INTDEV_LOG_WAKEUP
    MSG("<int_dev> wake up INT_IOC_WAIT_INT(%d)\n", int_num);
#endif
    if (ret) {
      return -EINTR;
    }
    if (BIT_ISSET(INTDEV_EVENT_CLOSE, info->receive_flag)) {
      return -EIDRM;
    }
    
    
    return 0;
    break;
    
  default:
    return -EINVAL;
    break;
  }
}

/*=======================================================================
 * interrupt handler
 *=======================================================================*/
static irqreturn_t
intdev_interrupt(int irq, void *dev_id)
{
  spin_lock(&intdev_lock);
  if( BIT_ISSET( INTDEV_STATE_USER_ENABLE, intr_info[irq].enable) ){
    MSG("Receive interrupt %d", irq);
    BIT_SET(INTDEV_STATE_INTR_DISABLE, intr_info[irq].enable);
    disable_irq_nosync(irq);
    BIT_SET(INTDEV_EVENT_INTR, intr_info[irq].receive_flag);
    wake_up_interruptible(&intr_info[irq].intr_wait_queue);
  }else{
    /* Not handling. */
    MSG("Receive interrupt %d(Not handling)", irq);
  }
  spin_unlock(&intdev_lock);
  return IRQ_HANDLED;
}

/* file operation structure */
static struct file_operations intdev_fops = {
  open:             intdev_local_open,
  release:          intdev_local_release,
  ioctl:   intdev_local_ioctl,
};

/*
 * int intdev_local_init( void )
 *-----------------------------------------------------------------------------
 * function: initialize module
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
int intdev_local_init( void )
{
  int i;
  int result;
  
  irqctl = (unsigned long)ioremap(IRQCTL, 4);
  if (!irqctl)
    goto ERROR_MALLOC;

  intr_info = kmalloc(sizeof(intr_info_t) * NR_IRQS, GFP_KERNEL);
  if (!intr_info)
    goto ERROR_MALLOC;
  
  for (i = 0; i < NR_IRQS; i++) {
    intr_info[i].fd_info = NULL;
    sprintf(intr_info[i].name, "irq_%d", i);
    BIT_ZERO(intr_info[i].enable);
    BIT_ZERO(intr_info[i].receive_flag);
    init_waitqueue_head(&intr_info[i].intr_wait_queue);
  }

  result = register_chrdev(INT_DEV_MAJOR, "intrdrv", &intdev_fops);
  if (0 > result)
    goto ERROR_REGDEV;

  return 0;
  
ERROR_REGDEV:
  kfree(intr_info);
  
ERROR_MALLOC:
  return -1;
}

/*
 * void intdev_local_exit( void )
 *-----------------------------------------------------------------------------
 * function: exit module
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
void intdev_local_exit( void )
{
  unregister_chrdev(INT_DEV_MAJOR, "intrdrv");
  kfree(intr_info);
  iounmap((void *)irqctl);
}

#endif /* INT_DEV_C */
